from flask import Flask, jsonify
import datetime
import random
import pytz

app = Flask(__name__)

def get_earth_time():
    """Returns current Earth time in ISO 8601 format."""
    now = datetime.datetime.now(pytz.timezone('Asia/Kolkata')) # Using local timezone as per earlier hint
    return now.isoformat()

def get_mars_time():
    """Calculates Coordinated Mars Time (MTC).
    Based on the formula:
    MSD = (JD_UT - 2451549.5) / 1.027491252 + 44796.0
    MTC = (MSD % 1) * 24
    """
    # 1. Get current UTC time
    now_utc = datetime.datetime.now(datetime.timezone.utc)
    
    # 2. Calculate Julian Date (UT)
    # 2440587.5 is the JD for 1970-01-01 00:00:00 UTC
    millis_since_epoch = now_utc.timestamp() * 1000
    jd_ut = 2440587.5 + (millis_since_epoch / 86400000.0)
    
    # 3. Calculate Mars Sol Date (MSD)
    msd = (jd_ut - 2451549.5) / 1.027491252 + 44796.0
    
    # 4. Calculate Coordinated Mars Time (MTC)
    mtc = (msd % 1) * 24
    
    # Format MTC as HH:MM:SS
    hours = int(mtc)
    minutes = int((mtc - hours) * 60)
    seconds = int(((mtc - hours) * 3600) % 60)
    
    return f"{hours:02d}:{minutes:02d}:{seconds:02d} MTC"

@app.route('/time', methods=['GET'])
def time_api():
    return jsonify({
        "earth_time": get_earth_time(),
        "mars_time": get_mars_time()
    })

# Global state for sensor simulation (Random Walk)
current_temp = 25.0
current_humidity = 50.0

@app.route('/dht22', methods=['GET'])
def dht22_api():
    global current_temp, current_humidity
    
    # Emulate DHT22 sensor values with Random Walk
    # Temp range typically -40 to 80 Celsius.
    # Humidity range 0-100%.
    
    # Generate random delta
    temp_delta = random.uniform(-3, 3)
    humidity_delta = random.uniform(-2, 2)
    
    # Apply delta
    current_temp += temp_delta
    current_humidity += humidity_delta
    
    # Clamp values to realistic ranges
    # Let's keep temperature somewhat stable for "room" conditions or just realistic outdoor
    # But allowing full range for testing if needed. 
    # Clamping to -20 to 50 for realistic "Earth-like" or "Mars-like" fluctuation simulation
    current_temp = max(-20.0, min(50.0, current_temp))
    current_humidity = max(0.0, min(100.0, current_humidity))
    
    return jsonify({
        "temperature": round(current_temp, 1),
        "humidity": round(current_humidity, 1)
    })

if __name__ == '__main__':
    # Host 0.0.0.0 is critical for local network access
    app.run(host='0.0.0.0', port=5000)
