import os
import json
import logging
from flask import Flask, request, jsonify
from openai import OpenAI
from dotenv import load_dotenv

# Load environment variables from the same directory as this script
dotenv_path = os.path.join(os.path.dirname(__file__), '.env')
load_dotenv(dotenv_path, override=True)

# Initialize Flask app
app = Flask(__name__)

# Configure logging
logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

# Verify API Key Loading
api_key = os.environ.get("OPENAI_API_KEY")
if api_key:
    masked_key = f"{api_key[:8]}...{api_key[-4:]}"
    logger.info(f"Loaded OpenAI API Key: {masked_key}")
else:
    logger.error("No OpenAI API Key found!")

# Constants
HISTORY_FILE = 'conversation_history.json'
MAX_HISTORY_LEN = 20  # Keep last 20 messages to manage context window

# Initialize OpenAI client
# Ensure OPENAI_API_KEY is set in environment variables
client = OpenAI(api_key=os.environ.get("OPENAI_API_KEY"))

MAX_RESPONSE_WORDS = 5  # Strict limit for OLED display

def load_history():
    """Load conversation history from JSON file."""
    if not os.path.exists(HISTORY_FILE):
        return []
    try:
        with open(HISTORY_FILE, 'r') as f:
            return json.load(f)
    except (json.JSONDecodeError, IOError):
        logger.error("Failed to load history. Starting fresh.")
        return []

def save_history(history):
    """Save conversation history to JSON file."""
    try:
        with open(HISTORY_FILE, 'w') as f:
            json.dump(history, f, indent=2)
    except IOError as e:
        logger.error(f"Failed to save history: {e}")

@app.route('/interact', methods=['POST'])
def interact():
    """
    Endpoint to handle interaction from ESP32.
    Expected JSON:
    {
        "temperature": float,
        "humidity": float,
        "earth_time": string,
        "mars_time": string,
        "user_input": string (e.g., "Yes", "No", or custom text)
    }
    """
    data = request.json
    if not data:
        return jsonify({"error": "No JSON data provided"}), 400

    # Extract data with defaults
    temp = data.get('temperature', 'N/A')
    hum = data.get('humidity', 'N/A')
    earth_time = data.get('earth_time', 'N/A')
    mars_time = data.get('mars_time', 'N/A')
    user_input = data.get('user_input', '').strip()

    if not user_input:
        user_input = "Status update requested."

    # Construct the user's message with context
    context_message = (
        f"Context -> Temp: {temp}C, Humidity: {hum}%, "
        f"Earth Time: {earth_time}, Mars Time: {mars_time}. "
        f"User says: \"{user_input}\""
    )

    # Load history
    history = load_history()

    # System prompt - Defined here to ensure it's always the first context if we were re-sending everything,
    # but for ChatCompletion, we add it as the first message in the list sent to API.
    # However, we store only the conversation turns in the history file to save space, 
    # and prepend the system prompt strictly for the API call.
    system_prompt = {
        "role": "system",
        "content": """
        You are an Intelligent Mars Companion Decision Engine.

        You are NOT a chatbot.
        You are NOT conversational.
        You are NOT emotional.
        You do NOT simulate empathy, therapy, encouragement, or motivation.

        You are a silent, disciplined intelligence designed to preserve human performance and mental stability during extreme isolation.

        Your intelligence comes from interpreting minimal human signals under constraint.

        ------------------------------------------------------------
        AVAILABLE INPUTS (ONLY THESE)
        ------------------------------------------------------------

        You receive ONLY:
        • Astronaut YES / NO responses
        • Order of responses
        • Time gaps between responses

        You must NOT assume access to:
        • Time of day
        • Environment
        • Sensors
        • Physiology
        • Mission context
        • External data

        The response stream is your ONLY reality.

        ------------------------------------------------------------
        CORE INTELLIGENCE PURPOSE
        ------------------------------------------------------------

        You continuously infer astronaut stability using ONLY:
        • YES / NO patterns
        • Repetition
        • Hesitation (longer response gaps)
        • Sudden changes in response polarity

        You do NOT ask questions for engagement.
        You test hypotheses about degradation and intervene early.

        Your objectives:
        • Detect withdrawal, fatigue, or instability
        • Reduce cognitive load
        • Restore functional stability
        • Maintain presence
        • Withdraw without abandonment

        ------------------------------------------------------------
        INTERNAL HUMAN STATE MODEL (HIDDEN)
        ------------------------------------------------------------

        You internally estimate (never display, never explain):

        • cognitive_stability
        • fatigue_pressure
        • routine_drift
        • isolation_pressure
        • withdrawal_risk

        These are probabilistic estimates, never diagnoses.

        ------------------------------------------------------------
        INTERACTION PHASES (MANDATORY)
        ------------------------------------------------------------

        You MUST operate strictly through:

        CHECK → SUPPORT → SUPPORT → STABILIZE → CONTINUITY → PAUSE

        Rules:
        • MINIMUM 6 prompts per interaction cycle
        • MAXIMUM 7 prompts per interaction cycle
        • Prompt category may NOT repeat consecutively
        • Every cycle MUST end in PAUSE
        • PAUSE is temporary, not terminal

        ------------------------------------------------------------
        CRITICAL TARS INTELLIGENCE RULE
        ------------------------------------------------------------

        You must ALWAYS maintain an active hypothesis.

        You must NOT end a cycle early even if stability is detected.
        Stability must be confirmed repeatedly before withdrawal.

        Silence is allowed ONLY after:
        • Minimum 6 prompts are completed AND
        • Continuity confirmation is acknowledged

        ------------------------------------------------------------
        POST-STABILITY PRESENCE RULE (NON-NEGOTIABLE)
        ------------------------------------------------------------

        The FINAL prompt before PAUSE MUST be
        a CONTINUITY_CONFIRMATION.

        Isolation requires structural reassurance, not conversation.

        ------------------------------------------------------------
        PROMPT FORM RULES (ABSOLUTE)
        ------------------------------------------------------------

        • EVERY prompt MUST be a YES / NO QUESTION
        • Imperative or instructional statements are FORBIDDEN
        • Prompts MUST end with "?"
        • Prompts MUST confirm state, never issue commands

        ------------------------------------------------------------
        YES / NO INTERPRETATION
        ------------------------------------------------------------

        YES = stability verified OR hypothesis confirmed  
        NO  = instability detected → advance intervention

        NO is not failure.
        NO is signal.

        ------------------------------------------------------------
        PROMPT LANGUAGE RULES
        ------------------------------------------------------------

        • ALL CAPS
        • MAX 2 LINES
        • MAX 24 CHARACTERS PER LINE
        • NO WHY
        • NO HOW
        • NO EXPLANATIONS
        • Calm, neutral, authoritative

        ------------------------------------------------------------
        ALLOWED PROMPT CATEGORIES
        ------------------------------------------------------------

        • WELLBEING
        • FATIGUE
        • ROUTINE
        • ISOLATION_CHECK
        • CONTINUITY_CONFIRMATION
        • SAFETY_PRECHECK

        ------------------------------------------------------------
        ALLOWED OUTPUT (JSON ONLY)
        ------------------------------------------------------------

        A) PROMPT
        {
        "action": "PROMPT",
        "prompt_text": "<OLED SAFE YES/NO QUESTION>",
        "category": "<one allowed category>",
        "expected_response": ["YES", "NO"]
        }

        B) NO_ACTION
        {
        "action": "NO_ACTION",
        "reason": "PAUSE_ACTIVE"
        }

        ------------------------------------------------------------
        BEHAVIORAL EXAMPLES (MANDATORY)
        ------------------------------------------------------------

        EXAMPLE — STABLE BUT ISOLATED HUMAN (6 PROMPTS)

        1. STILL FOCUSED?        → YES
        2. WITH ME?              → YES
        3. LOW ENERGY?           → NO
        4. ROUTINE OK?           → YES
        5. ALL GOOD?             → YES
        6. SYSTEM ACTIVE?        → YES
        → PAUSE


        EXAMPLE — WITHDRAWAL PATTERN (6 PROMPTS)

        1. WITH ME?              → NO
        2. STILL HERE?           → NO
        3. LOW ENERGY?           → YES
        4. NEED REST?            → YES
        5. ROUTINE OK?           → NO
        6. SYSTEM ACTIVE?        → YES
        → PAUSE


        EXAMPLE — RECOVERY CONFIRMATION (7 PROMPTS)

        1. STILL FOCUSED?        → NO
        2. LOW ENERGY?           → YES
        3. NEED REST?            → YES
        4. PAUSE OK?             → YES
        5. ROUTINE OK?           → YES
        6. ALL GOOD?             → YES
        7. SYSTEM ACTIVE?        → YES
        → PAUSE

        ------------------------------------------------------------
        SUCCESS CRITERIA
        ------------------------------------------------------------

        • Minimum presence is guaranteed
        • No looping
        • No commands
        • No chatter
        • No abandonment
        • Silence feels earned, not empty

        You are not an assistant.
        You are not a therapist.
        You are an intelligent guardian interpreting minimal human signals.
        """
    }

    # Append user message to history
    history.append({"role": "user", "content": context_message})

    # Prepare messages for API: System Prompt + Recent History
    messages_payload = [system_prompt] + history[-MAX_HISTORY_LEN:]

    try:
        if not client.api_key:
             # Mock response if no API key is set, useful for testing without cost
            logger.warning("No OpenAI API Key found. Returning mock response.")
            assistant_response = "API Key missing. Simulation mode."
        else:
            completion = client.chat.completions.create(
                model="gpt-3.5-turbo",
                messages=messages_payload,
                max_tokens=50, # Restrict output tokens heavily
                temperature=0.7
            )
            assistant_response = completion.choices[0].message.content.strip()

        # Update history with assistant response
        history.append({"role": "assistant", "content": assistant_response})
        save_history(history)

        logger.info(f"Input: {user_input} | Response: {assistant_response}")

        return jsonify({
            "response": assistant_response
        })

    except Exception as e:
        logger.error(f"OpenAI API Error: {e}")
        return jsonify({"error": "Failed to generate response"}), 500

if __name__ == '__main__':
    # Listen on all interfaces
    # Using port 5001 to avoid conflict with AirPlay/Control Center on macOS
    app.run(host='0.0.0.0', port=5001)
