from flask import Flask, request, jsonify, render_template, send_file
from datetime import datetime, timedelta
import json
import os
import urllib.request
import urllib.error
from openai import OpenAI
from dotenv import load_dotenv
import io

# Load environment variables
load_dotenv()

app = Flask(__name__)

# Initialize OpenAI Client
client = OpenAI(api_key=os.environ.get("OPENAI_API_KEY"))

# Global variable to store the latest data
latest_data = {
    "temperature": "N/A",
    "humidity": "N/A",
    "earth_time": "Waiting for data...",
    "mars_time": "Waiting for data...",
    "conversation_history": []
}

# Timestamp for last sensor data fetch
last_sensor_fetch_time = None

# Path to conversation history file
BASE_DIR = os.path.dirname(os.path.abspath(__file__))
PARENT_DIR = os.path.dirname(BASE_DIR)
HISTORY_FILE_PATH = os.path.join(PARENT_DIR, 'LLM_Interaction_APIs', 'conversation_history.json')

def fetch_api_data(url):
    try:
        with urllib.request.urlopen(url) as response:
            return json.loads(response.read().decode())
    except Exception as e:
        print(f"Error fetching from {url}: {e}")
        return None

@app.route('/')
def index():
    return render_template('index.html')

@app.route('/tts', methods=['POST'])
def tts_endpoint():
    try:
        data = request.json
        text = data.get('text')
        voice = data.get('voice', 'alloy')

        if not text:
            return jsonify({"error": "No text provided"}), 400

        response = client.audio.speech.create(
            model="tts-1",
            voice=voice,
            input=text
        )

        # Stream the response directly back
        return send_file(
            io.BytesIO(response.content),
            mimetype="audio/mpeg",
            as_attachment=False,
            download_name="speech.mp3"
        )

    except Exception as e:
        print(f"TTS Error: {e}")
        return jsonify({"error": str(e)}), 500

@app.route('/webhook', methods=['POST'])
def webhook():
    global latest_data
    try:
        data = request.json
        if not data:
            return jsonify({"status": "error", "message": "No JSON data received"}), 400
        
        # Update latest data with received values, keeping defaults if keys are missing
        latest_data["temperature"] = data.get("temperature", latest_data["temperature"])
        latest_data["humidity"] = data.get("humidity", latest_data["humidity"])
        latest_data["earth_time"] = data.get("earth_time", latest_data["earth_time"])
        latest_data["mars_time"] = data.get("mars_time", latest_data["mars_time"])
        latest_data["conversation_history"] = data.get("conversation_history", latest_data["conversation_history"])
        
        print(f"Received data: {latest_data}")
        return jsonify({"status": "success", "message": "Data received"}), 200
    except Exception as e:
        print(f"Error processing webhook: {e}")
        return jsonify({"status": "error", "message": str(e)}), 500

@app.route('/data', methods=['GET'])
def get_data():
    global latest_data, last_sensor_fetch_time
    
    # 1. Fetch Time Data (Always fetch to keep clocks smooth)
    time_data = fetch_api_data('http://localhost:5000/time')
    if time_data:
        latest_data['earth_time'] = time_data.get('earth_time', latest_data['earth_time'])
        latest_data['mars_time'] = time_data.get('mars_time', latest_data['mars_time'])

    # 2. Fetch Sensor Data (Throttled to every 5 seconds)
    current_time = datetime.now()
    if last_sensor_fetch_time is None or (current_time - last_sensor_fetch_time) >= timedelta(seconds=5):
        sensor_data = fetch_api_data('http://localhost:5000/dht22')
        if sensor_data:
            latest_data['temperature'] = sensor_data.get('temperature', latest_data['temperature'])
            latest_data['humidity'] = sensor_data.get('humidity', latest_data['humidity'])
        last_sensor_fetch_time = current_time

    # 3. Read specific Conversation History file (Always fetch for real-time chat)
    try:
        if os.path.exists(HISTORY_FILE_PATH):
            with open(HISTORY_FILE_PATH, 'r') as f:
                history_content = json.load(f)
                latest_data['conversation_history'] = history_content
        else:
             print(f"Warning: History file not found at {HISTORY_FILE_PATH}")
    except Exception as e:
        print(f"Error reading history file: {e}")

    return jsonify(latest_data)

if __name__ == '__main__':
    # Run on all network interfaces to be accessible by ESP32
    app.run(host='0.0.0.0', port=5002, debug=True)
